document.addEventListener('DOMContentLoaded', function() {
  // DOM元素
  const loginSection = document.getElementById('login-section');
  const userSection = document.getElementById('user-section');
  const rechargeSection = document.getElementById('recharge-section');
  const messageDiv = document.getElementById('message');

  const phoneInput = document.getElementById('phone');
  const passwordInput = document.getElementById('password');
  const captchaCodeInput = document.getElementById('captcha-code');
  const captchaImage = document.getElementById('captcha-image');
  const refreshCaptchaBtn = document.getElementById('refresh-captcha');

  const loginBtn = document.getElementById('login-btn');
  const registerBtn = document.getElementById('register-btn');
  const getAnswerBtn = document.getElementById('get-answer-btn');
  const rechargeBtn = document.getElementById('recharge-btn');
  const logoutBtn = document.getElementById('logout-btn');
  const backToMainBtn = document.getElementById('back-to-main');
  const userPhoneSpan = document.getElementById('user-phone');
  const coinCountSpan = document.getElementById('coin-count');
  const coinOptions = document.querySelectorAll('.coin-option');

  const paymentInfo = document.getElementById('payment-info');
  const paymentIdSpan = document.getElementById('payment-id');
  const paymentAmountSpan = document.getElementById('payment-amount');
  const checkPaymentBtn = document.getElementById('check-payment-btn');
  const cancelPaymentBtn = document.getElementById('cancel-payment-btn');

  // 配置与状态
  const API_BASE = 'https://ks.ld0972.com/api';
  let currentCaptchaId = null;
  let captchaLoading = false;
  let captchaTimer = null;
  const CAPTCHA_VALID_TIME = 90000; // 验证码有效期1.5分钟（缩短有效期减少过期概率）

  let currentPaymentId = null;
  let paymentPollingInterval = null;

  // 初始化
  checkLoginStatus();
  loadCaptcha(); // 首次加载验证码

  // 事件监听
  refreshCaptchaBtn.addEventListener('click', loadCaptcha);
  captchaImage.addEventListener('click', loadCaptcha);
  loginBtn.addEventListener('click', handleLogin);
  registerBtn.addEventListener('click', handleRegister);
  getAnswerBtn.addEventListener('click', getAnswer);
  rechargeBtn.addEventListener('click', showRechargeSection);
  logoutBtn.addEventListener('click', logout);
  backToMainBtn.addEventListener('click', showUserSection);
  checkPaymentBtn.addEventListener('click', () => currentPaymentId && checkPaymentStatus(currentPaymentId));
  cancelPaymentBtn.addEventListener('click', cancelPayment);

  coinOptions.forEach(option => {
    option.addEventListener('click', function() {
      initiateRecharge(this.getAttribute('data-amount'));
    });
  });

  // 验证码核心功能 - 修复自动刷新问题
  async function loadCaptcha(forceRefresh = true) {
    // 防止重复请求
    if (captchaLoading) return;

    // 清除旧的定时器
    if (captchaTimer) {
      clearTimeout(captchaTimer);
      captchaTimer = null;
    }

    // 显示加载状态
    captchaLoading = true;
    captchaImage.src = 'data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMTAwIiBoZWlnaHQ9IjQwIiB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciPjxwYXRoIGQ9Ik01MCwyMEMzMC41LDIwIDE1LDMyLjYgMTUsNTBDMTUsNjcuNCAzMC41LDgwIDUwLDgwQzc5LjUsODAgOTUsNjcuNCA5NSw1MEM5NSwzMi42IDc5LjUsMjAgNTAsMjBaIiBmaWxsPSIjZmZmZmZmIiBmaWxsLW9wYWNpdHk9IjAuMiIvPjxwYXRoIGQ9Ik01MCwyNUM2NC40LDI1IDc1LDM1LjYgNzUsNTBDNzUsNjQuNCA2NC40LDc1IDUwLDc1QzM1LjYsNzUgMjUsNjQuNCAyNSw1MEMyNSwzNS42IDM1LjYsMjUgNTAsMjVaIiBmaWxsPSIjZmZmZmZmIi8+PGNpcmNsZSBjeD0iNTAiIGN5PSI1MCIgcj0iMTUiIGZpbGw9IiNmNWY3ZmYiIGZpbGwtb3BhY2l0eT0iMC41Ii8+PGNpcmNsZSBjeD0iNTAiIGN5PSI1MCIgcj0iMTAiIGZpbGw9IiNmMGYwZjAiIGZpbGwtb3BhY2l0eT0iMC44Ii8+PGNpcmNsZSBjeD0iNTAiIGN5PSI1MCIgcj0iNSIgZmlsbD0iI2ZmZmZmZiIvPjxwYXRoIGQ9Ik01MCwzMEg1MFY0MEg1MFYzMCIgZmlsbD0iIzAwMDAwMCIvPjwvc3ZnPg==';
    refreshCaptchaBtn.disabled = true;
    refreshCaptchaBtn.textContent = '加载中...';

    try {
      // 关键修复：添加时间戳防止缓存，确保每次请求都是新的验证码
      const timestamp = new Date().getTime();
      const response = await fetch(`${API_BASE}/captcha?_t=${timestamp}`, {
        method: 'GET',
        cache: 'no-store',
        headers: {
          'Cache-Control': 'no-cache, no-store, must-revalidate',
          'Pragma': 'no-cache'
        }
      });

      if (!response.ok) {
        throw new Error(`HTTP错误: ${response.status}`);
      }

      const data = await response.json();

      // 验证返回数据有效性
      if (!data || !data.captcha_id || !data.image) {
        throw new Error('验证码数据格式错误');
      }

      // 更新验证码状态
      currentCaptchaId = data.captcha_id;
      captchaImage.src = `data:image/png;base64,${data.image}`;
      captchaCodeInput.value = '';

      // 设置有效期定时器 - 提前10秒刷新
      captchaTimer = setTimeout(() => {
        showMessage('验证码即将过期，正在刷新...');
        loadCaptcha();
      }, CAPTCHA_VALID_TIME - 10000);

    } catch (error) {
      console.error('加载验证码失败:', error);
      showMessage('验证码加载失败，请点击刷新', true);
      currentCaptchaId = null;
    } finally {
      // 恢复状态
      captchaLoading = false;
      refreshCaptchaBtn.disabled = false;
      refreshCaptchaBtn.textContent = '换一张';
    }
  }

  // 登录处理
  async function handleLogin() {
    // 基础验证
    const phone = phoneInput.value.trim();
    const password = passwordInput.value;
    const captchaCode = captchaCodeInput.value.trim();

    if (!phone || !password || !captchaCode) {
      showMessage('请填写完整信息', true);
      return;
    }

    if (!/^1[3-9]\d{9}$/.test(phone)) {
      showMessage('请输入有效的手机号', true);
      return;
    }

    // 验证码有效性检查
    if (!currentCaptchaId) {
      showMessage('请先获取验证码', true);
      loadCaptcha();
      return;
    }

    try {
      showMessage('登录中...');
      const response = await fetch(`${API_BASE}/login`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          phone,
          password,
          captcha_id: currentCaptchaId,
          captcha_code: captchaCode
        })
      });

      // 处理非JSON响应
      const contentType = response.headers.get('content-type');
      if (!contentType || !contentType.includes('application/json')) {
        throw new Error('服务器响应格式错误');
      }

      const data = await response.json();

      if (response.ok && data.token) {
        // 登录成功
        await chrome.storage.local.set({ token: data.token });
        userPhoneSpan.textContent = phone;
        coinCountSpan.textContent = data.coins || 0;
        showUserSection();
        showMessage('登录成功');
        resetLoginForm();
        loadCaptcha(); // 登录成功后刷新验证码
      } else {
        // 登录失败
        const errorMsg = data.message || '登录失败';
        showMessage(errorMsg, true);

        // 验证码相关错误，强制刷新
        if (errorMsg.includes('验证码') && (errorMsg.includes('过期') || errorMsg.includes('不存在') || errorMsg.includes('错误'))) {
          loadCaptcha();
        }
      }
    } catch (error) {
      console.error('登录错误:', error);
      showMessage('登录失败，请重试', true);
    }
  }

  // 注册处理
  async function handleRegister() {
    // 基础验证
    const phone = phoneInput.value.trim();
    const password = passwordInput.value;
    const captchaCode = captchaCodeInput.value.trim();

    if (!phone || !password || !captchaCode) {
      showMessage('请填写完整信息', true);
      return;
    }

    if (!/^1[3-9]\d{9}$/.test(phone)) {
      showMessage('请输入有效的手机号', true);
      return;
    }

    if (password.length < 6) {
      showMessage('密码长度不能少于6位', true);
      return;
    }

    // 验证码有效性检查
    if (!currentCaptchaId) {
      showMessage('请先获取验证码', true);
      loadCaptcha();
      return;
    }

    try {
      showMessage('注册中...');
      const response = await fetch(`${API_BASE}/register`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          phone,
          password,
          captcha_id: currentCaptchaId,
          captcha_code: captchaCode
        })
      });

      // 处理非JSON响应
      const contentType = response.headers.get('content-type');
      if (!contentType || !contentType.includes('application/json')) {
        throw new Error('服务器响应格式错误');
      }

      const data = await response.json();

      if (response.ok) {
        // 注册成功
        showMessage('注册成功，请登录');
        // resetLoginForm();
        loadCaptcha(); // 注册成功后刷新验证码
      } else {
        // 注册失败
        const errorMsg = data.message || '注册失败';
        showMessage(errorMsg, true);

        // 验证码相关错误，强制刷新
        if (errorMsg.includes('验证码') && (errorMsg.includes('过期') || errorMsg.includes('不存在') || errorMsg.includes('错误'))) {
          loadCaptcha();
        }
      }
    } catch (error) {
      console.error('注册错误:', error);
      showMessage('注册失败，请重试', true);
    }
  }

  // 辅助函数：重置登录表单
  function resetLoginForm() {
    phoneInput.value = '';
    passwordInput.value = '';
    captchaCodeInput.value = '';
  }

  // 检查登录状态
  async function checkLoginStatus() {
    const tokenData = await chrome.storage.local.get('token');
    const token = tokenData.token;

    if (token) {
      try {
        const response = await fetch(`${API_BASE}/user-info`, {
          headers: { 'Authorization': `Bearer ${token}` }
        });

        if (response.ok) {
          const userData = await response.json();
          userPhoneSpan.textContent = userData.phone || '';
          coinCountSpan.textContent = userData.coins || 0;
          showUserSection();
          return;
        }
      } catch (error) {
        console.error('检查登录状态失败:', error);
      }
    }

    // 未登录或验证失败
    showLoginSection();
  }

  // 页面切换函数
  function showLoginSection() {
    stopPaymentPolling();
    hidePaymentInfo();
    loginSection.classList.remove('hidden');
    userSection.classList.add('hidden');
    rechargeSection.classList.add('hidden');
  }

  function showUserSection() {
    stopPaymentPolling();
    hidePaymentInfo();
    loginSection.classList.add('hidden');
    userSection.classList.remove('hidden');
    rechargeSection.classList.add('hidden');
  }

  function showRechargeSection() {
    stopPaymentPolling();
    hidePaymentInfo();
    loginSection.classList.add('hidden');
    userSection.classList.add('hidden');
    rechargeSection.classList.remove('hidden');
  }

  // 消息提示
  function showMessage(text, isError = false) {
    messageDiv.textContent = text;
    messageDiv.classList.remove('hidden');
    messageDiv.classList.toggle('error', isError);

    // 3秒后自动隐藏
    setTimeout(() => {
      if (messageDiv.textContent === text) {
        hideMessage();
      }
    }, 3000);
  }

  function hideMessage() {
    messageDiv.classList.add('hidden');
  }

  // 退出登录
  async function logout() {
    await chrome.storage.local.remove('token');
    showLoginSection();
    showMessage('已退出登录');
  }

   // 获取答案
  async function getAnswer() {
    const token = await chrome.storage.local.get('token');

    if (!token || !token.token) {
      showMessage('请先登录', true);
      return;
    }

    showMessage('下载答案需等待1-2分钟||正在获取题目并生成答案...');

    try {
      // 获取当前标签页内容
      const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });

      // 执行内容脚本获取页面文本
      const results = await chrome.scripting.executeScript({
        target: { tabId: tab.id },
        function: getPageTextContent
      });

      if (results && results[0] && results[0].result) {
        const pageContent = results[0].result;

        // 发送到API处理
        const response = await fetch(`${API_BASE}/get-answer`, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${token.token}`
          },
          body: JSON.stringify({ content: pageContent })
        });

        const data = await response.json();

        if (response.ok) {
          // 下载答案文件
          chrome.downloads.download({
            url: data.download_url,
            filename: `exam_answer_${Date.now()}.txt`
          });

          // 更新元宝数量
          const userResponse = await fetch(`${API_BASE}/user-info`, {
            headers: {
              'Authorization': `Bearer ${token.token}`
            }
          });

          if (userResponse.ok) {
            const userData = await userResponse.json();
            coinCountSpan.textContent = userData.coins;
          }

          showMessage('答案已生成并开始下载');
        } else {
          showMessage(data.message || '获取答案失败', true);
        }
      } else {
        showMessage('无法获取页面内容', true);
      }
    } catch (error) {
      showMessage('获取答案失败: ' + error.message, true);
    }
  }
  // 充值相关函数
  async function initiateRecharge(amount) {
    const tokenData = await chrome.storage.local.get('token');
    const token = tokenData.token;

    if (!token) {
      showMessage('请先登录', true);
      return;
    }

    try {
      const response = await fetch(`${API_BASE}/create-payment`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${token}`
        },
        body: JSON.stringify({ amount })
      });

      const data = await response.json();

      if (response.ok && data.payment_id && data.pay_url) {
        showPaymentInfo(data.payment_id, amount);
        chrome.windows.create({ url: data.pay_url });
        startPaymentPolling(data.payment_id);
        showMessage('请在新页面完成支付');
      } else {
        showMessage(data.message || '创建订单失败', true);
      }
    } catch (error) {
      console.error('充值错误:', error);
      showMessage('创建订单失败，请重试', true);
    }
  }

 // 检查支付状态
  async function checkPaymentStatus(paymentId) {
    const token = await chrome.storage.local.get('token');

    try {
      const response = await fetch(`${API_BASE}/payment-status/${paymentId}`, {
        headers: {
          'Authorization': `Bearer ${token.token}`
        }
      });

      const data = await response.json();

      if (data.status === 'paid') {
        stopPaymentPolling();
        showMessage('支付成功，元宝已到账');

        // 更新用户信息
        const userResponse = await fetch(`${API_BASE}/user-info`, {
          headers: {
            'Authorization': `Bearer ${token.token}`
          }
        });

        if (userResponse.ok) {
          const userData = await userResponse.json();
          coinCountSpan.textContent = userData.coins;
        }

        // 隐藏支付信息
        hidePaymentInfo();

      } else if (data.status === 'failed') {
        stopPaymentPolling();
        showMessage('支付失败或已取消', true);
        hidePaymentInfo();
      }
      // pending状态继续轮询
    } catch (error) {
      showMessage('检查支付状态失败: ' + error.message, true);
    }
  }

  function startPaymentPolling(paymentId) {
    stopPaymentPolling();

    const check = async () => {
      const success = await checkPaymentStatus(paymentId);
      if (!success) {
        paymentPollingInterval = setTimeout(check, 5000);
      }
    };

    check();
  }

  function stopPaymentPolling() {
    if (paymentPollingInterval) {
      clearTimeout(paymentPollingInterval);
      paymentPollingInterval = null;
    }
  }

  function showPaymentInfo(paymentId, amount) {
    paymentIdSpan.textContent = paymentId;
    paymentAmountSpan.textContent = amount;
    paymentInfo.classList.remove('hidden');
    currentPaymentId = paymentId;
  }

  function hidePaymentInfo() {
    paymentInfo.classList.add('hidden');
    currentPaymentId = null;
  }

  function cancelPayment() {
    stopPaymentPolling();
    hidePaymentInfo();
    showMessage('支付已取消');
    showRechargeSection();
  }

  // 页面内容获取
  function getPageTextContent() {
     // 获取页面所有文本内容
    const bodyText = document.body.innerText;
    return bodyText;
  }
});



// 放在文件的末尾或现有 DOMContentLoaded 事件监听之后
document.addEventListener('DOMContentLoaded', function() {
  // 确保在现有初始化逻辑之后执行
  checkForUpdates();

  // 绑定标题点击事件（如果之前没有）
  const updateHeader = document.getElementById('update-header');
  if (updateHeader) {
    updateHeader.addEventListener('click', function() {
      window.open('https://ks.ld0972.com', '_blank'); // 官网下载页
    });
  } else {
    console.error('未找到 update-header 元素');
  }
});

// 检查更新函数
async function checkForUpdates() {
  // 获取当前版本（从 manifest.json 中）
  const currentVersion = chrome.runtime.getManifest().version;
  const updateNotice = document.getElementById('update-notice');
  const updateHeader = document.getElementById('update-header');

  // 检查元素是否存在
  if (!updateNotice || !updateHeader) {
    console.error('更新提示所需的DOM元素不存在');
    return;
  }

  try {
    // 调用后端接口（使用实际的API地址）
    const response = await fetch('https://ks.ld0972.com/api/check-update', {
      method: 'GET',
      headers: {
        'Content-Type': 'application/json'
      }
    });

    if (!response.ok) {
      throw new Error(`HTTP 错误: ${response.status}`);
    }

    const data = await response.json();

    // 版本对比（使用之前提到的版本比较函数）
    if (data.latest_version && isNewVersion(currentVersion, data.latest_version)) {
      updateNotice.classList.remove('hidden');
      updateHeader.textContent = `考试宝（有新版本 v${data.latest_version}）`;
    }
  } catch (error) {
    console.log('检查更新失败:', error);
    // 静默失败，不影响主功能
  }
}

// 版本比较辅助函数（确保能正确比较 1.10 > 1.2 等情况）
function isNewVersion(current, latest) {
  const currentParts = current.split('.').map(Number);
  const latestParts = latest.split('.').map(Number);

  for (let i = 0; i < Math.max(currentParts.length, latestParts.length); i++) {
    const curr = currentParts[i] || 0;
    const lat = latestParts[i] || 0;
    if (lat > curr) return true;
    if (lat < curr) return false;
  }
  return false;
}